/* Copyright (C) 2000-2002 Lavtech.com corp. All rights reserved.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA 
*/

#include "udm_config.h"
#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <string.h>
#include <errno.h>
#include <regex.h>

#include "udm_common.h"
#include "udm_spell.h"
#include "udm_proto.h"
#include "udm_url.h"
#include "udm_parser.h"
#include "udm_conf.h"
#include "udm_log.h"
#include "udm_hrefs.h"
#include "udm_robots.h"
#include "udm_utils.h"
#include "udm_host.h"
#include "udm_server.h"
#include "udm_alias.h"
#include "udm_filter.h"
#include "udm_search_tl.h"
#include "udm_env.h"
#include "udm_mimetype.h"
#include "udm_stopwords.h"
#include "udm_guesser.h"
#include "udm_unicode.h"
#include "udm_synonym.h"
#include "udm_vars.h"
#include "udm_db.h"
#include "udm_agent.h"

/****************************  Load Configuration **********************/

int UdmSearchMode(const char * mode){
	if(!mode)return(UDM_MODE_ALL);
	if(!strcmp(mode,"all"))return(UDM_MODE_ALL);
	if(!strcmp(mode,"any"))return(UDM_MODE_ANY);
	if(!strcmp(mode,"bool"))return(UDM_MODE_BOOL);
	if(!strcmp(mode,"phrase"))return(UDM_MODE_PHRASE);
	return(UDM_MODE_ALL);
}


int UdmMatchMode(const char * mode){
	if(!mode)return(UDM_MATCH_FULL);
	if(!strcmp(mode,"wrd"))return(UDM_MATCH_FULL);
	if(!strcmp(mode,"full"))return(UDM_MATCH_FULL);
	if(!strcmp(mode,"beg"))return(UDM_MATCH_BEGIN);
	if(!strcmp(mode,"end"))return(UDM_MATCH_END);
	if(!strcmp(mode,"sub"))return(UDM_MATCH_SUBSTR);
	return(UDM_MATCH_FULL);
}


int UdmFollowType(const char * follow){
	if(!follow)return UDM_FOLLOW_UNKNOWN;
	if(!strcasecmp(follow,"no"))return(UDM_FOLLOW_NO);
	if(!strcasecmp(follow,"page"))return(UDM_FOLLOW_NO);
	if(!strcasecmp(follow,"yes"))return(UDM_FOLLOW_PATH);
	if(!strcasecmp(follow,"path"))return(UDM_FOLLOW_PATH);
	if(!strcasecmp(follow,"site"))return(UDM_FOLLOW_SITE);
	if(!strcasecmp(follow,"world"))return(UDM_FOLLOW_WORLD);
	return(UDM_FOLLOW_UNKNOWN);
}

void UdmWeightFactorsInit(const char *wf, int *res){
	size_t len;
	int sn;
	
	for(sn=0;sn<256;sn++){
		res[sn]=1;
	}
	
	len=strlen(wf);
	if((len>0)&&(len<256)){
		const char *sec;
			
		for(sec=wf+len-1;sec>=wf;sec--){
			res[len-(sec-wf)]=UdmHex2Int(*sec);
		}
	}
}

#define IF_NUM_PARAM(src,name,dst,def) \
if(!UDM_STRNCASECMP(src,name)){if(sscanf(src+strlen(name),"%d",&(dst))!=1)dst=def;}

#define IF_TIME_PARAM(src,name,dst,def) \
if (!UDM_STRNCASECMP(src,name)){ \
	if ((dst=Udm_dp2time_t(src+strlen(name)))==-1) \
		dst=def; \
}

#define IF_BOOL_PARAM(src,name,dst,def) \
if(!UDM_STRNCASECMP(src,name)){ \
char xxx[1024]; \
if(sscanf(src+strlen(name),"%s",xxx)==1){ \
	if(!UDM_STRNCASECMP(xxx,"yes"))dst=1; \
	else	dst=0; \
}else \
	dst=def; \
}


__INDLIB__  int UdmLoadConfig(UDM_ENV * Conf,const char *conf_name,int config_level,int load_flags){
	int i=0; 		/* Line number */
	FILE *config;		/* File struct */
	char str[1024]="";	/* Unsafe copy - will be used in strtok	*/
	char str1[1024]="";	/* To concatenate lines			*/
	char ostr[1024]="";	/* Safe copy of original  config string	*/
	UDM_HREF Href;
	
	Conf->errcode=0;
	Conf->errstr[0]=0;

	if(config_level==0){ /* Do some initialization */
		Conf->Spells.nspell=0;
		Conf->csrv=(UDM_SERVER*)malloc(sizeof(UDM_SERVER));
		UdmServerInit(Conf->csrv);
		
		/* I have removed it to allow adding of URLs  */
		/* Passed in -i with -u command line args     */
		/* But it seems that UnloadConf() is required */
		/* REMOVED      UdmFreeHrefs(); */
		Conf->lcs=UdmGetCharSet("latin1");
	}
	
	/* Open config file */
	if(!(config=fopen(conf_name,"r"))){
		sprintf(Conf->errstr,"Error: can't open config file '%s': %s",conf_name, strerror(errno));
		Conf->errcode=1;
		return UDM_ERROR;
	}
	
	/*  Read lines and parse */
	while(fgets(str1,sizeof(str1),config)){
		char *end;
		i++;
		if(!str1[0])continue;
		end=str1+strlen(str1)-1;
		while((end>=str1)&&(*end=='\r'||*end=='\n'||*end==' ')){
			*end=0;if(end>str1)end--;
		}
		if(!str1[0])continue;
		if(str1[0]=='#')continue;

		if(*end=='\\'){
			*end=0;strcat(str,str1);
			continue;
		}
		strcat(str,str1);
		strcpy(str1,"");
		strcpy(ostr,str);

		if(!UDM_STRNCASECMP(str,"DBAddr")){
			UdmVarListAddStr(&Conf->Vars,"DBAddr",UdmTrim(str+6," \t\r\n"));
		}else
		if(!UDM_STRNCASECMP(str,"VarDir")){
			sprintf(Conf->vardir,"%s%s",UdmTrim(str+6," \r\n\t"),UDMSLASHSTR);
		}else
		if(!UDM_STRNCASECMP(str,"DataDir")){
			sprintf(Conf->vardir,"%s%s",UdmTrim(str+6," \r\n\t"),UDMSLASHSTR);
		}else
		if(!UDM_STRNCASECMP(str,"LogdAddr")){
			UdmVarListReplaceStr(&Conf->Vars,"LogdAddr",UdmTrim(str+8," \r\n\t"));
		}else
		if(!UDM_STRNCASECMP(str,"Listen")){
			UdmVarListReplaceStr(&Conf->Vars,"Listen",UdmTrim(str+6," \r\n\t"));
		}else
		if(!UDM_STRNCASECMP(str,"URLDAddr")){
			UdmVarListReplaceStr(&Conf->Vars,"URLDAddr",UdmTrim(str+8," \r\n\t"));
		}else
		if(!UDM_STRNCASECMP(str,"StoredAddr")){
			UdmVarListReplaceStr(&Conf->Vars,"StoredAddr",UdmTrim(str+10," \r\n\t"));
		}else
		if(!UDM_STRNCASECMP(str,"LocalCharset")){
			char *local_charset=UdmTrim(str+13," \r\n\t");
			if(!(Conf->lcs = UdmGetCharSet(local_charset))){
				sprintf(Conf->errstr,"Error in config file '%s' line %d: charset '%s' is not supported\n",conf_name,i,local_charset);
				Conf->errcode=1;
				return UDM_ERROR;
			}
			UdmVarListReplaceStr(&Conf->Vars,"LocalCharset",local_charset);
		}else
		if(!UDM_STRNCASECMP(str, "BrowserCharset")){
			char *browser_charset=UdmTrim(str + 15, " \r\n\t");
			if(!(Conf->bcs = UdmGetCharSet(browser_charset))){
				sprintf(Conf->errstr,"Error in config file '%s' line %d: charset '%s' is not supported\n",conf_name,i,browser_charset);
				Conf->errcode=1;
				return UDM_ERROR;
			}
			UdmVarListReplaceStr(&Conf->Vars,"BrowserCharset",browser_charset);
		}else
		if(!UDM_STRNCASECMP(str,"UseRemoteContentType")){
			char *arg=UdmTrim(str+20," \t");
			UdmVarListAddStr(&Conf->Vars,"UseRemoteContentType",arg);
		}else
		if(!UDM_STRNCASECMP(str,"UseCRC32URLID")){
			char *arg=UdmTrim(str+13," \t");
			UdmVarListAddStr(&Conf->Vars,"UseCRC32URLID",arg);
		}else
		if(!UDM_STRNCASECMP(str,"NewsExtensions")){
			char *arg=UdmTrim(str+14," \t");
			UdmVarListAddStr(&Conf->Vars,"NewsExtensions",arg);
		}else
		if(!UDM_STRNCASECMP(str,"CrossWords")){
			char *arg=UdmTrim(str+10," \t");
			UdmVarListAddStr(&Conf->Vars,"CrossWords",arg);
		}else
		if(!UDM_STRNCASECMP(str,"Disallow")){
			if(UdmFilterAddStr(Conf,str,UDM_METHOD_DISALLOW))
				return UDM_ERROR;
		}else	
		if(!UDM_STRNCASECMP(str,"Allow")){
			if(UdmFilterAddStr(Conf,str,UDM_METHOD_GET))
				return UDM_ERROR;
		}else
		if(!UDM_STRNCASECMP(str,"CheckMP3Only")){
			if(UdmFilterAddStr(Conf,str,UDM_METHOD_CHECKMP3ONLY))
				return UDM_ERROR;
		}else
		if(!UDM_STRNCASECMP(str,"CheckMP3")){
			if(UdmFilterAddStr(Conf,str,UDM_METHOD_CHECKMP3))
				return UDM_ERROR;
		}else
		if(!UDM_STRNCASECMP(str,"CheckOnly")){
			if(UdmFilterAddStr(Conf,str,UDM_METHOD_HEAD))
				return UDM_ERROR;
		}else
		if(!UDM_STRNCASECMP(str,"HrefOnly")){
			if(UdmFilterAddStr(Conf,str,UDM_METHOD_HREFONLY))
				return UDM_ERROR;
		}else
		if(!UDM_STRNCASECMP(str,"AddType")){
			char *mime_type=NULL,*lt,*ttok;
			int flags=0;

			ttok=UdmGetStrToken(str+7,&lt);
			while(ttok){
				if(!strcasecmp(ttok,"regex"))
					flags|=UDM_MIME_REGEX;
				else
				if(!strcasecmp(ttok,"regexp"))
					flags|=UDM_MIME_REGEX;
				else
				if(!strcasecmp(ttok,"string"))
					flags&=~UDM_MIME_REGEX;
				else
				if(!strcasecmp(ttok,"case"))
					flags|=UDM_MIME_CS;
				else
				if(!strcasecmp(ttok,"nocase"))
					flags&=~UDM_MIME_CS;
				else
				if(!mime_type)
					mime_type=ttok;
				else{
					if(UdmMimeTypeAdd(Conf,mime_type,ttok,flags))
						return UDM_ERROR;
				}
				ttok=UdmGetStrToken(NULL,&lt);
			}
		}else
		if(!UDM_STRNCASECMP(str,"RemoteCharset")){
			const char *cs=UdmTrim(str+13," \r\n\t");
			if(!UdmGetCharSet(cs)){
				sprintf(Conf->errstr,"Error in config file '%s' line %d: charset '%s' is not supported\n",conf_name,i,cs);
				Conf->errcode=1;
				return UDM_ERROR;
			}
			UdmVarListReplaceStr(&Conf->csrv->Vars,"Default-Charset",cs);
		}else
		if(!UDM_STRNCASECMP(str,"ProxyAuthBasic")){
			char *basic_auth=NULL;
			if(sscanf(str+14,"%s",str1)){
				basic_auth=(char*)malloc(BASE64_LEN(strlen(str1)));
				udm_base64_encode(str1,basic_auth,strlen(str1));
			}
			UdmVarListReplaceStr(&Conf->csrv->ExtraHeaders,"Proxy-Authorization",basic_auth?basic_auth:"");
			UDM_FREE(basic_auth);
		}else
		if(!UDM_STRNCASECMP(str,"Proxy")){
			str1[0]='\0';
			if(sscanf(str+5,"%s",str1)&&strcmp(str1,"")){
			}
			UdmVarListReplaceStr(&Conf->csrv->ExtraHeaders,"Proxy",str1);
		}else
		if(!UDM_STRNCASECMP(str,"Category")){
			const char *arg=UdmTrim(str+8," \r\n\t");
			UdmVarListReplaceStr(&Conf->csrv->Vars,"Category",arg);
		}else
		if(!UDM_STRNCASECMP(str,"Tag")){
			const char *arg=UdmTrim(str+3," \r\n\t");
			UdmVarListReplaceStr(&Conf->csrv->Vars,"Tag",arg);
		}else
		if(!UDM_STRNCASECMP(str,"HTTPHeader")){
			char * arg;
			arg=UdmTrim(str+11," \r\n\t");
			if(sscanf(arg,"%[^\n\r]s",str1)&&strcmp(str1,"")){
				char *val;
				if((val=strchr(str1,':'))){
					*val++='\0';
					val=UdmTrim(val," \t");
					UdmVarListAddStr(&Conf->ExtraHeaders,str1,val);
				}
			}
		}else
		if(!UDM_STRNCASECMP(str,"SyslogFacility")){
			if(sscanf(str+14,"%s",str1)&&strcmp(str1,""))
				UdmVarListAddStr(&Conf->Vars,"SyslogFacility",str1);
		}else
		if(!UDM_STRNCASECMP(str,"AuthBasic")){
			
			if(sscanf(str+9,"%s",str1)){
				char	*ch, *basic_auth, *user = NULL, *passwd = NULL;
				basic_auth=(char*)malloc(BASE64_LEN(strlen(str1)));
				udm_base64_encode(str1,basic_auth,strlen(str1));
				if ((ch=strchr(str1, ':'))){
					user=(char*)malloc((size_t)(ch-str1+1));
					snprintf(user, (size_t)(ch-str1+1), "%s", str1);
					passwd=(char*)malloc(strlen(ch)+1);
					sprintf(passwd, "%s", ch+1);
				}
				UdmVarListReplaceStr(&Conf->csrv->ExtraHeaders,"Authorization",basic_auth);
				UDM_FREE(basic_auth);
				UDM_FREE(user);
				UDM_FREE(passwd);
			}else{
				UdmVarListReplaceStr(&Conf->csrv->ExtraHeaders,"Authorization","");
			}
		}else
		if(!UDM_STRNCASECMP(str,"HTDBList")){
			const char *arg=UdmTrim(str+8," \r\n\t");
			UdmVarListReplaceStr(&Conf->csrv->Vars,"HTDBList",arg);
		}else
		if(!UDM_STRNCASECMP(str,"HTDBDoc")){
			const char *arg=UdmTrim(str+8," \r\n\t");
			UdmVarListReplaceStr(&Conf->csrv->Vars,"HTDBDoc",arg);
		}else
		if(!UDM_STRNCASECMP(str,"Mime")){
			char *mime_arg[4],*lt,*mtok;
			int argn=0;

			mtok=UdmGetStrToken(str+4,&lt);
			for(;(mtok)&&(argn<4);argn++){
				mime_arg[argn]=mtok;
				mtok=UdmGetStrToken(NULL,&lt);	
			}
			if(argn!=3){
				sprintf(Conf->errstr,"Error in config file '%s' line %d too %s arguments for Mime command\n",conf_name,i,(argn<3?"few":"many"));
				Conf->errcode=1;
				return UDM_ERROR;
			}else{
				UDM_PARSER P;
				P.from_mime=mime_arg[0];
				P.to_mime=mime_arg[1];
				P.cmd=mime_arg[2];
				UdmParserAdd(&Conf->Parsers,&P);
			}
		}else
		if(!UDM_STRNCASECMP(str,"AliasProg")){
			char * aprog,* lt;
			if((aprog=UdmGetStrToken(str+9,&lt)))
				UdmVarListAddStr(&Conf->Vars,"AliasProg",aprog);
		}else
		if(!UDM_STRNCASECMP(str,"Alias")){
			char * ttok, * lt;
			UDM_ALIAS Alias;
			
			bzero(&Alias,sizeof(Alias));
			Alias.match.match_type=UDM_MATCH_BEGIN;
			
			ttok=UdmGetStrToken(str+5,&lt);
			while(ttok){
				if(!strcasecmp(ttok,"regex"))
					Alias.match.match_type=UDM_MATCH_REGEX;
				else
				if(!strcasecmp(ttok,"regexp"))
					Alias.match.match_type=UDM_MATCH_REGEX;
				else
				if(!strcasecmp(ttok,"case"))
					Alias.match.case_sense=0;
				else
				if(!strcasecmp(ttok,"nocase"))
					Alias.match.case_sense=UDM_MATCH_ICASE;
				else
				if(!Alias.match.pattern){
					Alias.match.pattern=ttok;
				}else{
					char err[120]="";
					Alias.replace=ttok;
					if(UdmAliasListAdd(&Conf->Aliases,&Alias,err,sizeof(err))){
						snprintf(Conf->errstr,sizeof(Conf->errstr)-1,"Error in config file '%s' line %d: %s: %s",conf_name,i,err,ostr);
						Conf->errcode=1;
						return UDM_ERROR;
					}
				}
				ttok=UdmGetStrToken(NULL,&lt);
			}
			if(!Alias.replace){
				snprintf(Conf->errstr,sizeof(Conf->errstr)-1,"Error in config file '%s' line %d: too few arguments: %s",conf_name,i,ostr);
				Conf->errcode=1;
				return UDM_ERROR;
			}
		}else
		if(!UDM_STRNCASECMP(str,"ReverseAlias")){
			char * ttok, * lt;
			UDM_ALIAS Alias;
			
			bzero(&Alias,sizeof(Alias));
			Alias.match.match_type=UDM_MATCH_BEGIN;
			
			ttok=UdmGetStrToken(str+12,&lt);
			while(ttok){
				if(!strcasecmp(ttok,"regex"))
					Alias.match.match_type=UDM_MATCH_REGEX;
				else
				if(!strcasecmp(ttok,"regexp"))
					Alias.match.match_type=UDM_MATCH_REGEX;
				else
				if(!strcasecmp(ttok,"case"))
					Alias.match.case_sense=0;
				else
				if(!strcasecmp(ttok,"nocase"))
					Alias.match.case_sense=UDM_MATCH_ICASE;
				else
				if(!Alias.match.pattern){
					Alias.match.pattern=ttok;
				}else{
					char err[120]="";
					Alias.replace=ttok;
					if(UdmAliasListAdd(&Conf->ReverseAliases,&Alias,err,sizeof(err))){
						snprintf(Conf->errstr,sizeof(Conf->errstr)-1,"Error in config file '%s' line %d: %s: %s",conf_name,i,err,ostr);
						Conf->errcode=1;
						return UDM_ERROR;
					}
				}
				ttok=UdmGetStrToken(NULL,&lt);
			}
			if(!Alias.replace){
				snprintf(Conf->errstr,sizeof(Conf->errstr)-1,"Error in config file '%s' line %d: too few arguments: %s",conf_name,i,ostr);
				Conf->errcode=1;
				return UDM_ERROR;
			}
		}else
		if(!UDM_STRNCASECMP(str,"LangMapFile")){
			if(load_flags&UDM_FLAG_LOAD_LANGMAP){
				int res;
				char * arg;
				char filename[1024];

				arg=UdmTrim(str+11," \r\n\t");
				if(arg[0]=='/'){
					sprintf(filename,"%s",arg);
				}else{
					sprintf(filename,"%s/%s",UDM_CONF_DIR,arg);
				}
				if((res=UdmLoadLangMapFile(Conf,filename))){
					return UDM_ERROR;
				}
			}
		}else
		if(!UDM_STRNCASECMP(str,"DefaultLang")){
			const char *arg=UdmTrim(str+11," \r\n\t");
			UdmVarListReplaceStr(&Conf->csrv->Vars,"Default-Language",arg);
		}else
		if(!UDM_STRNCASECMP(str,"ServerTable")){
			if(load_flags&UDM_FLAG_ADD_SERV){
				char		*addr=UdmTrim(str+11," \r\n\t");
				int		res;
				UDM_AGENT	Agent;
				void		*db=UdmDBInit(NULL);
				
				UdmDBSetAddr(db,addr,UDM_OPEN_MODE_READ);
				UdmAgentInit(&Agent,Conf,0);
				res=UdmSrvAction(&Agent,&Conf->Servers,UDM_SRV_ACTION_TABLE,db);
				UdmAgentFree(&Agent);
				UdmDBFree(db);
				if(res!=UDM_OK)return res;
			}
		}else
		if(!UDM_STRNCASECMP(str,"Server")||!UDM_STRNCASECMP(str,"Realm")||!UDM_STRNCASECMP(str,"Subnet")){
			int	follow;
			char	*cmd, *lt;
			
			if(!UDM_STRNCASECMP(str,"Server")){
				Conf->csrv->match_type=UDM_SERVER_SUBSTR|UDM_SERVER_MATCH;
			}else
			if(!UDM_STRNCASECMP(str,"Subnet")){
				Conf->csrv->match_type=UDM_SERVER_SUBNET|UDM_SERVER_MATCH;
				Conf->Servers.have_subnets=1;
			}else{
				Conf->csrv->match_type=UDM_SERVER_STRING|UDM_SERVER_MATCH;
			}
			
			/* Store follow value */
			follow=Conf->csrv->Spider.follow;
			
			cmd=strtok_r(str," \t\r\n",&lt);
			
			while((cmd=strtok_r(NULL," \t\r\n",&lt))){
				if(!strcasecmp(cmd,"no"))Conf->csrv->Spider.follow=UDM_FOLLOW_NO;
				else
				if(!strcasecmp(cmd,"page"))Conf->csrv->Spider.follow=UDM_FOLLOW_NO;
				else
				if(!strcasecmp(cmd,"yes"))Conf->csrv->Spider.follow=UDM_FOLLOW_PATH;
				else
				if(!strcasecmp(cmd,"path"))Conf->csrv->Spider.follow=UDM_FOLLOW_PATH;
				else
				if(!strcasecmp(cmd,"site"))Conf->csrv->Spider.follow=UDM_FOLLOW_SITE;
				else
				if(!strcasecmp(cmd,"world"))Conf->csrv->Spider.follow=UDM_FOLLOW_WORLD;
				else
				if(!strcasecmp(cmd,"nocase"))Conf->csrv->match_type&=~UDM_SERVER_CS;
				else
				if(!strcasecmp(cmd,"case"))Conf->csrv->match_type|=UDM_SERVER_CS;
				else
				if(!strcasecmp(cmd,"match"))Conf->csrv->match_type|=UDM_SERVER_MATCH;
				else
				if(!strcasecmp(cmd,"nomatch"))Conf->csrv->match_type&=~UDM_SERVER_MATCH;
				else
				if(!strcasecmp(cmd,"string"))UDM_SET_SRV_TYPE(Conf->csrv->match_type,UDM_SERVER_STRING);
				else
				if(!strcasecmp(cmd,"regex"))UDM_SET_SRV_TYPE(Conf->csrv->match_type,UDM_SERVER_REGEX);
				else{
					if(!Conf->csrv->url)
						Conf->csrv->url=strdup(cmd);
					else
					if(!Conf->csrv->alias)
						Conf->csrv->alias=strdup(cmd);
					else{
						Conf->errcode=1;
						sprintf(Conf->errstr,"'%s' line %d: too many argiments: '%s'",conf_name,i,cmd);
						return UDM_ERROR;
					}
				}
			}
			if(UdmServerAdd(Conf,Conf->csrv)){
				char * s_err;
				Conf->errcode=1;
				s_err=strdup(Conf->errstr);
				sprintf(Conf->errstr,"'%s' line %d: %s",conf_name,i,s_err);
				free(s_err);
				UDM_FREE(Conf->csrv->url);
				return UDM_ERROR;
			}
			if((UDM_SRV_TYPE(Conf->csrv->match_type)==UDM_SERVER_SUBSTR)&&(Conf->csrv->url[0])&&(load_flags&UDM_FLAG_ADD_SERV)){
				bzero(&Href,sizeof(Href));
				Href.url=Conf->csrv->url;
				Href.method=UDM_METHOD_GET;
				UdmHrefListAdd(&Conf->Hrefs,&Href);
			}
			UDM_FREE(Conf->csrv->url);
			UDM_FREE(Conf->csrv->alias);
			/* Restore follow value */
			Conf->csrv->Spider.follow=follow;
			
		}else
		if(!UDM_STRNCASECMP(str,"Follow")){
			if(sscanf(str+6,"%s",str1)!=1)
				continue;
			Conf->csrv->Spider.follow=UdmFollowType(str1);
			if(Conf->csrv->Spider.follow==UDM_FOLLOW_UNKNOWN){
				Conf->errcode=1;
				sprintf(Conf->errstr,"'%s' line %d: Unknown follow type: %s",conf_name,i,str1);
				return UDM_ERROR;
			}
		}
		else	IF_BOOL_PARAM(str,"Robots",Conf->csrv->Spider.use_robots,1)
		else	IF_BOOL_PARAM(str,"DetectClones",Conf->csrv->Spider.use_clones,1)
		else	IF_BOOL_PARAM(str,"Index",Conf->csrv->Spider.index,1)
				
		else	IF_NUM_PARAM(str,"MaxNetErrors",Conf->csrv->Spider.max_net_errors,UDM_MAXNETERRORS)
		else	IF_TIME_PARAM(str,"NetErrorDelayTime",Conf->csrv->Spider.net_error_delay_time,UDM_DEFAULT_NET_ERROR_DELAY_TIME)
		else	IF_TIME_PARAM(str,"ReadTimeOut",Conf->csrv->Spider.read_timeout,UDM_READ_TIMEOUT)
		else	IF_TIME_PARAM(str,"DocTimeOut",Conf->csrv->Spider.doc_timeout,UDM_DOC_TIMEOUT)
		else	IF_TIME_PARAM(str,"Period",Conf->csrv->Spider.period,UDM_DEFAULT_REINDEX_TIME)
		else	IF_TIME_PARAM(str, "HoldBadHrefs", Conf->bad_since_time, UDM_DEFAULT_BAD_SINCE_TIME)
		else	IF_NUM_PARAM(str,"MaxHops",Conf->csrv->Spider.maxhops,0)
   		
		else	IF_NUM_PARAM(str,"IspellCorrectFactor",Conf->WordParam.correct_factor,1)
		else	IF_NUM_PARAM(str,"IspellIncorrectFactor",Conf->WordParam.incorrect_factor,1)
		else	IF_NUM_PARAM(str,"NumberFactor",Conf->WordParam.number_factor,1)
		else	IF_NUM_PARAM(str,"AlnumFactor",Conf->WordParam.alnum_factor,1)
		else	IF_NUM_PARAM(str,"MinWordLength",Conf->WordParam.min_word_len,1)
		else	IF_NUM_PARAM(str,"MaxWordLength",Conf->WordParam.max_word_len,32)
		else
		if(!UDM_STRNCASECMP(str,"MaxDocSize")){
			int sz=atoi(str+10);
			UdmVarListAddInt(&Conf->Vars,"MaxDocSize",sz?sz:UDM_MAXDOCSIZE);
		}else
		if(!UDM_STRNCASECMP(str,"DocMemCacheSize")){
			int sz=atof(str+15)*1024*1024;
			UdmVarListAddInt(&Conf->Vars,"DocMemCacheSize",sz);
		}else
		if(!UDM_STRNCASECMP(str,"ForceIISCharset1251")){
			char *arg=UdmTrim(str+19," \t");
			UdmVarListAddStr(&Conf->Vars,"force1251",arg);
		}else
		if(!UDM_STRNCASECMP(str,"URL")){/* Should be after URLWeight etc */
			if(load_flags&UDM_FLAG_ADD_SERV){
				if(sscanf(str+3,"%s",str1)&&strcmp(str1,"")){
					char al[UDM_URLSIZE*2];
					UDM_SERVER * Srv;
					
					if((Srv=UdmServerFind(&Conf->Servers,str1,"",al))){
						Href.url=str1;
						Href.tag=NULL;	/*Srv->tag; FIXME */
						Href.category=NULL;/*Srv->category;FIXME*/
						Href.referrer=0;
						Href.hops=0;
						Href.stored=0;
						Href.method=UDM_METHOD_GET;
						UdmHrefListAdd(&Conf->Hrefs,&Href);
					}
				}
			}
		}else
		if(!UDM_STRNCASECMP(str,"Affix")){
			char lang[33];
			char charset[41];
			if(load_flags&&UDM_FLAG_SPELL){
				if(3==sscanf(str+5,"%32s%40s%s",lang,charset,str1)){
					if(*str1=='/')strcpy(str,str1);
					else	sprintf(str,"%s/%s",UDM_CONF_DIR,str1);
					if(UdmImportAffixes(Conf,lang,charset,str)){
						Conf->errcode=1;
						sprintf(Conf->errstr,"Can't load affix :%s",str);
						return UDM_ERROR;
					}
				}
				else {
					Conf->errcode=1;
					sprintf(Conf->errstr,"Error in config file '%s' line %d:%s",conf_name,i,str);
					return UDM_ERROR;
				}
			}
		}else
		if(!UDM_STRNCASECMP(str,"Spell")){
			char lang[33];
			char charset[41];
			if(load_flags&&UDM_FLAG_SPELL){
				if(3==sscanf(str+5,"%32s%40s%s",lang,charset,str1)){
					if(*str1=='/')strcpy(str,str1);
					else	sprintf(str,"%s/%s",UDM_CONF_DIR,str1);
					/* -----  0 or 1 ?  */
					if(UdmImportDictionary(Conf,lang,charset,str,0,"")){
					        Conf->errcode=1;
					        sprintf(Conf->errstr,"Can't load dictionary :%s",str);
						return UDM_ERROR;
					}
				} else {
					Conf->errcode=1;
					sprintf(Conf->errstr,"Error in config file '%s' line %d:%s",conf_name,i,str);
					return UDM_ERROR;
				}
			}
		}else
		if(!UDM_STRNCASECMP(str, "IspellUsePrefixes")) {
		  char sel[8];
		  int val = 1;
		  sscanf(str+17, "%4s", sel);
		  if (!UDM_STRNCASECMP(sel, "no")) {
		    val = 0;
		  }
		  UdmVarListAddInt(&Conf->Vars, "IspellUsePrefixes", val);
		}else
		if(!UDM_STRNCASECMP(str, "StoredFiles")) {
		  int val = 0x10000;
		  sscanf(str+11, "%4d", &val);
		  UdmVarListAddInt(&Conf->Vars, "StoredFiles", val);
		}else
		if(!UDM_STRNCASECMP(str, "OptimizeInterval")) {
		  int val = 600;
		  sscanf(str + 16, "%4d", &val);
		  UdmVarListAddInt(&Conf->Vars, "OptimizeInterval", val);
		}else
		if(!UDM_STRNCASECMP(str, "OptimizeRatio")) {
		  int val = 5;
		  sscanf(str + 13, "%4d", &val);
		  UdmVarListAddInt(&Conf->Vars, "OptimizeRatio", val);
		}else
		if(!UDM_STRNCASECMP(str, "TrackQuery")) {
		  char sel[8];
		  sscanf(str+10, "%4s", sel);
		  UdmVarListAddStr(&Conf->Vars, "TrackQuery", sel);
		}else
		if(!UDM_STRNCASECMP(str, "Cache")) {
		  char sel[8];
		  sscanf(str+5, "%4s", sel);
		  UdmVarListAddStr(&Conf->Vars, "Cache", sel);
		}else
		if(!UDM_STRNCASECMP(str,"Synonym")){
			char * fname;
			char aname[256]="";
			
			if(load_flags&&UDM_FLAG_SPELL){
				fname=UdmTrim(str+7," \t\r\n");
				if(*fname=='/')strcpy(aname,fname);
				else	sprintf(aname,"%s/%s",UDM_CONF_DIR,fname);
				if(UdmSynonymListLoad(Conf,aname)){
					Conf->errcode=1;
					return UDM_ERROR;
				}
			}
		}else
		if(!UDM_STRNCASECMP(str,"Include")){
			if((sscanf(str+7,"%s",str1))&&(config_level<5)){
				if(*str1=='/')strcpy(str,str1);
				else	sprintf(str,"%s/%s",UDM_CONF_DIR,str1);
				UdmLoadConfig(Conf,str,config_level+1,load_flags);
				if(Conf->errcode)return UDM_ERROR;
			}
		}else
		if(!UDM_STRNCASECMP(str,"MirrorRoot")){
			const char *arg=UdmTrim(str+10," \t\r\n");
			if(*arg=='/')
				strcpy(str1,arg);
			else
				sprintf(str1,"%s/%s",UDM_VAR_DIR,arg);
			UdmVarListReplaceStr(&Conf->csrv->Vars,"MirrorRoot",str1);
		}else
		if(!UDM_STRNCASECMP(str,"MirrorHeadersRoot")){
			const char *arg=UdmTrim(str+17," \t\r\n");
			if(*arg=='/')
				strcpy(str1,arg);
			else
				sprintf(str1,"%s/%s",UDM_VAR_DIR,arg);
			UdmVarListReplaceStr(&Conf->csrv->Vars,"MirrorHeadersRoot",str1);
		}else
		if(!UDM_STRNCASECMP(str,"MirrorPeriod")){
			char	*arg=UdmTrim(str+12," \t\r\n");
			int	tm=Udm_dp2time_t(arg);
			UdmVarListReplaceInt(&Conf->csrv->Vars,"MirrorPeriod",tm);
		}else
		if(!UDM_STRNCASECMP(str,"StopwordFile")){
			if(sscanf(str+12,"%s",str1)){
				if(*str1=='/')strcpy(str,str1);
				else    sprintf(str,"%s/%s",UDM_CONF_DIR,str1);
				if(UdmStopListLoad(Conf,str)){
					Conf->errcode=1;
					return UDM_ERROR;
				}
			} else {
				Conf->errcode=1;
				sprintf(Conf->errstr,"Error in config file '%s' line %d:%s",conf_name,i,str);
				return UDM_ERROR;
			}
		}else
		if(!UDM_STRNCASECMP(str,"Limit")){
		        char * sc, * nm;
			if(sscanf(str+5, "%s", str1)){
			  if((sc = strchr(str1,':'))){
			    *sc='\0';sc++;
			    nm=(char*)malloc(strlen(str1)+8);
			    sprintf(nm,"Limit-%s",str1);
			    UdmVarListAddStr(&Conf->Vars, nm, sc);
			    free(nm);
			  }
			} else {
			  Conf->errcode=1;
			  sprintf(Conf->errstr,"Error in config file '%s' line %d:%s",conf_name,i,str);
			  return UDM_ERROR;
			}
		}else
		if(!UDM_STRNCASECMP(str,"Section")){
			char *name, *lt, *sec=NULL, *mlen=NULL;
			name=UdmGetStrToken(str+7,&lt);
			if(name)sec=UdmGetStrToken(NULL,&lt);
			if(sec)mlen=UdmGetStrToken(NULL,&lt);
			if(name&&sec){
				UDM_VAR S;
				
				bzero(&S,sizeof(S));
				S.name=name;
				S.section=atoi(sec);
				S.maxlen= mlen ? atoi(mlen) : 0;
				UdmVarListAdd(&Conf->Sections,&S);
			}else{
				snprintf(Conf->errstr,sizeof(Conf->errstr),"Too few arguments in config file '%s' line %d",conf_name,i);
				Conf->errcode=1;
				return UDM_ERROR;
			}
		}else{
			Conf->errcode=1;
			sprintf(Conf->errstr,"Error in config file '%s' line %d: %s",conf_name,i,str);
			return UDM_ERROR;
		}
		str[0]=0;
	}
	fclose(config);
	
	/* On level0 : Free some variables, prepare others, etc */
	if(config_level==0){
		const char	*dbaddr=NULL;
		
		/* Add one virtual server if we want FollowOutside */
		/* or DeleteNoServer no	*/
		UDM_FREE(Conf->csrv->url);
		if(Conf->csrv->Spider.follow==UDM_FOLLOW_WORLD){
			Conf->csrv->match_type=UDM_SERVER_SUBSTR|UDM_SERVER_MATCH;
			UdmServerAdd(Conf,Conf->csrv);
		}
		
		/* Removed due to new Realm command             */
		/* Servers/Realms now found in appearence order */
		/* UdmSortServers(Conf);                        */
		
		UdmServerFree(Conf->csrv);
		UDM_FREE(Conf->csrv);
		
		/* Sort Aliases for binary search            */
		/* This is incorrect - aliases must          */
		/* be found in the order of their appearance */
		/* UdmSortAliases(Conf); */
		
		/* Sort ispell dictionay if it has been loaded */
		
		if(Conf->Spells.nspell) {
			UdmSortDictionary(&Conf->Spells);
			UdmSortAffixes(&Conf->Affixes, &Conf->Spells);
		}
		
		/* Sort synonyms */
		UdmSynonymListSort(&Conf->Synonyms);
		
		for( i=0 ; i< Conf->ExtraHeaders.nvars ; i++)
			if(!strcasecmp(Conf->ExtraHeaders.Var[i].name,"User-Agent"))
				goto agent;
		
		UdmVarListAddStr(&Conf->ExtraHeaders,"User-Agent",UDM_USER_AGENT "/" VERSION);
agent:
		
#ifdef HAVE_SQL
		if(!dbaddr)dbaddr=UdmVarListFindStr(&Conf->Vars,"DBAddr","mysql://localhost/mnogosearch/");
#endif
#ifdef HAVE_FILES
		if(!dbaddr)dbaddr=UdmVarListFindStr(&Conf->Vars,"DBAddr", "file:" UDM_VAR_DIR UDMSLASHSTR);
#endif
		if(!dbaddr)dbaddr=UdmVarListFindStr(&Conf->Vars,"DBAddr", "searchd://localhost/");
		
		if(UDM_OK!=UdmDBSetAddr(Conf->db,dbaddr,UDM_OPEN_MODE_WRITE)){
			sprintf(Conf->errstr,"Invalid DBAddr: '%s'",dbaddr);
			Conf->errcode=1;
			return UDM_ERROR;
		}
	}
	return UDM_OK;
}
